package Mail::Eudora;
# see POD documentation at end

$VERSION = 0.69;
$debug = 0;

print STDERR "\tusing Mail::Eudora v. $VERSION\n";

require 5.000;
require Exporter;

@ISA = qw(Exporter);
@EXPORT = qw(get_messages get_messages_from_file parse_message append_message);

$error = "";
$Eudora_date_header = 'X-eudora-date: ';
$address_rx = '\b(([\w-]+(?:\.[\w-]+)*)\@([\w-]+(?:\.[\w-]+)+))\b'; # v. 0.1 - where full=$1, user=$2, domain=$3
$sep_rx = 'From \?{3}\@\?{3} '; # separator regex, to match literall 'From ???@??? '

sub get_messages {
# return list of messages in Eudora .mbx file
# takes one lvalue argument with all messages

	my $mbx;
	if (ref($_[0])) { $mbx =  $_[0] }
    else            { $mbx = \$_[0] };

	my @messages = split /$sep_rx/, $$mbx;
	shift @messages; # 1st is empty, since $mbx starts with a separator

	foreach (@messages) {
		$_ = $Eudora_date_header . $_; # put the date after 'From ???@??? ' into a header format
	}

	return @messages;

} # end sub get_messages

sub get_messages_from_file {
# try a more efficient method to...
# return list of messages in Eudora .mbx file
# takes one argument with the file name

	my $F = shift;
	my @messages;

	local($/);
	$/ = "From ???\@??? "; #Eudora msg. separator is literally 'From ???@??? '

	unless (open(F, "<$F")) {
		$error .= "Couldn't open $F : $!\n";
		return 0;
	}
	
	my $tmp = <F>; # 1st is just the 1st 'From ???@??? '
	$tmp = <F>;
	chomp($tmp);
	$messages[0] = $Eudora_date_header . $tmp;
	
	while (<F>) {
		chomp;
		push @messages, "$Eudora_date_header$_";
	}

	unless (close F) {$error .= "Couldn't close $F : $!\n"; warn "$error"} 
	
	return @messages;
} # end sub get_messages_from_file

sub parse_message {
# return a hash with headers and body of message
	my $msg = shift;
	print STDERR "No message!\n" if($debug && !$msg);
	my (%message, $header, $body, @headers, $error_count);
	if ($msg =~ /^(.+?\n)\n(.+)/so) {
		($header, $body) = ($1, $2);
	}
	else {
		($header, $body) = ($msg, "");
	}
	print STDERR "No header!\n" if($debug && !$header);
	print STDERR "No Body!\n" if($debug && !$body);
	$message{Body} = $body;
	@headers = split /^([\w\-]+): */mo, $header;
	print STDERR "No \@headers!\n" if($debug && !@headers);
	shift @headers; # shift 'cause split makes 1st element empty!
	while (@headers) {
		$header = ucfirst lc(shift @headers);
		unless (exists $message{$header}) { $message{$header} = "" };
		$message{$header} .= shift @headers;
	}
	return %message;
} # sub parse_message

sub append_message {
# append a message obtained with parse_message to an Eudora .mbx message file
	my $FILE = shift;
	my %message = @_;
	my $header;
	
	unless (open FILE, ">>$FILE") {
		$error = "Couldn't open $FILE -- $!\n";
		return undef;
	}
	
	print FILE 'From ???@??? ', $message{'X-eudora-date'};
	delete $message{'X-eudora-date'};
	
	# order some of the headers to print them in the correct position
	my @order = qw(X-Persona Received Sender);
	
	# reform Received headers
	if (exists $message{Received}) {
		$message{Received} =~ s/\n(\S)/\nReceived: $1/go;
	}

	# print first headers
	foreach $header (@order) {
		if (exists $message{$header}) {
			print FILE $header, ": ", $message{$header};
			delete $message{$header};
		}
	}
	
	# now extract body and print all other headers
	my $body =$message{Body};
	delete $message{Body};

	foreach $header (keys %message) {
		print FILE $header, ": ", $message{$header};
	}
	
	# add custom header and then print body
	print FILE "X-hack: Hacked into mailbox by Mail::Eudora v. $Mail::Eudora::VERSION\n";
	
	print FILE "\n", $body, "\n";

	unless (close FILE) {
		$error = "Couldn't close $FILE -- $!\n";
		return 0;
	}
	return 1;
} # sub parse_message

1;
__END__


=head1 NAME

Mail::Eudora version 0.69 - Manipulate Eudora mailbox files


=head1 SYNOPSIS

  use Mail::Eudora;
  print "Using Mail::Eudora version $Mail::Eudora::VERSION";
  @messages = get_messages_from_file($filename);
  # or read an Eudora .mbx file into $mbx, and then
  # @messages = get_messages($mbx);
  %message_24 = parse_message(@messages[24]);
  append_message($FILE, %message_24) or warn "$Mail::Eudora::error";


=head1 DESCRIPTION

Mail::Eudora provides a few subs to manipulate Eudora .mbx mailbox
files. It doesn't handle the .toc index files (Qualcomm doesn't disclose
it's .toc file structure, and says they keep changing it). This means
that if you use	append_message, you have to delete the .toc file, and
loose all Eudora proprietary information (labels, status, etc...).
	
It exports the following subs to your namespace:

  get_messages

  get_messages_from_file

  parse_message

  append_message

=head1 USAGE DETAILS

=over 4

=item get_messages()

C<@messages = get_messages($mbx);>

- takes a string containing any number of messages from an 
  Eudora .mbx mailbox file

- returns an array containing all single messages.

A new header is created in each message:

X-eudora-date: with the date Eudora added when it got the message.
When you use append_message, this header is replaced back with the original
Eudora message separator: 'From ???@??? ' 


=item get_messages_from_file()

C<@messages = get_messages_from_file($FILE);>

same as get_messages(), except it takes a string containing the
Eudora .mbx mailbox filename instead of a string with the messages

On big files, this seems much faster than get_messages.

=item parse_message()

C<%message = parse_message(@messages[24]);>

- takes a string containing one message (usually an element of the
  array returned by get_messages)

- returns a hash where keys are header names plus a 'Body' key and
  values are what you expect: values...

$subject = $message{Subject};

$body = $message{Body};

$x_info = $message{'X-info'}; 

etc...

$message{Received} contains all Received: headers. They are restored
to separate headers when you use append_message.

case of headers is normalized using ucfirst( lc $header ). So even if
the original header was "X-Info", now it is "X-info".

=item append_message()

C<append_message($FILE, %message) or warn "$Mail::Eudora::error";>

- takes a string containing a file name and a hash containing one
  message - usually obtained with parse_message(@messages).

- returns
  1 on success,
  undef on an error opening the file,
  0 on an error closing the file (in case you care to check for
  the difference).
  If there was an error, it's now in $Mail::Eudora::error.

append_message reformats %message into the correct Eudora format
and appends it to $FILE.

It also adds an X-hack: header, so you have	a trace that this message
was added by your script.

=item package variables

The following variables are not exported, but you can still access
them with their full name:

=over 4

=item $Mail::Eudora::error

Error on a file open or close in append_message


=item $Mail::Eudora::VERSION

The package version number

=back

=back

=head1 AUTHOR

	Milivoj Ivkovic <mi@alma.ch> or <ivkovic@csi.com>


=head1 NOTES

	You can use this freely.

	I would appreciate a short (or long) e-mail note if you do.
	And of course, bug-reports and/or improvements are welcome.

	Last revision: 14.07.98. Latest version should be available at
	http://alma.ch/perl

=cut